<?php
namespace X\Accounting;

use X\ETrace\System;

class SessionChecksumError extends \X\ETrace\Notification {
}
class SessionHacked extends \X\ETrace\Notification {
}

/**
 * Session Manager
 *
 *    Session struct:
 *    t    = type(int)
 *    s    = session_id
 *    cc    = crypto code (спец код с возможностю прокрутки назад и вперед.)
 *    cs    = crypto cheksum
 *    a    = activation_time
 *    u    = user_id
 *    hs    = is_https(bool)
 *
 *    crypto cheksum:
 *        cheksum = ((( a & ( ! s ) ) | cc ) ^ u ) >> (t ^ op1)
 *
 *    crypto code:
 *        code = rand(0,time());
 *        next_code = (code >> op2) ^ op3
 *        prev_code =    (code ^ op3) << op2
 */
class Session extends \X\Security\Crypto\IDEA {
	//protected function BitwiseCROR($v, $c)
	//protected function BitwiseCROL($v, $c)
	use \X\Tool\BitwiseCyclicShift;
	/**
	 * @var array
	 */
	protected $session_data = null;
	/**
	 * @var mixed
	 */
	protected $session = null;
	/**
	 * @var string
	 */
	protected $session_name = "s";

	/**
	 * @var array
	 */
	protected $param_crypto = [2, 6, 4];

	/**
	 * @param string $key          - password
	 * @param string $name         - collumn name
	 * @param array  $param_crypto - [int op1, int op2, int op3]
	 */
	public function __construct($key, $name = "s", $param_crypto = false) {
		parent::__construct($key);
		if ($param_crypto) {
			$this->param_crypto = $param_crypto;
		}
		$this->session_name = $name;
	}

	/**
	 * @return mixed
	 */
	public function get_session_data() {
		return $this->session_data;
	}

	public function get_system_bits() {
		switch (PHP_INT_SIZE) {
			case 4:
				return 32;
				break;
			case 8:
				return 64;
				break;
			default:
				throw new \X\ETrace\System("OS bits PROBLEM");
		}
	}

	/**
	 * @return mixed
	 */
	public function get_session() {
		return $this->session;
	}

	protected function read_session() {
		$In            = new \X_Input();
		$this->session = $In->CookieValue($this->session_name, false) ?: $In->Request($this->session_name, "")->string();
		if (strlen($this->session) > 0) {
			if (is_string($SessionString = $this->decrypt_b64($this->session))) {
				if (is_array($session_data = $this->explode(gzuncompress($SessionString)))) {
					if (isset($session_data["cs"])) {
						$checksum_valid = false;
						if (isset($session_data["b"])) {
							if ($session_data["b"] == 32 && $session_data["cs"] == $this->crypto_checksum($session_data, 32)) $checksum_valid = true;
							if ($session_data["b"] == 64 && $session_data["cs"] == $this->crypto_checksum($session_data, 64)) $checksum_valid = true;
						} else {
							if ($session_data["cs"] == $this->crypto_checksum($session_data, 32)) {
								$checksum_valid    = true;
								$session_data["b"] = 32;
							}
							if ($session_data["cs"] == $this->crypto_checksum($session_data, 64)) {
								$checksum_valid    = true;
								$session_data["b"] = 64;
							}
						}
						if ($checksum_valid) {
							$this->session_data = $session_data;
							return true;
						} else {
							throw new SessionChecksumError("Checksum Error", ["in_function" => get_defined_vars(), "in_class" => $this->session]);
						}
					}
				}
			}
		}
		return false;
	}

	/**
	 * @param $session_data
	 */
	protected function make_session($session_data) {
		if ( ! isset($session_data["cc"])) {
			$session_data["cc"] = $this->crypto_code_new();
		}
		$session_data["b"]  = $this->get_system_bits();
		$session_data["cs"] = $this->crypto_checksum($session_data, $session_data["b"]);
		$this->session_data = $session_data;
		return $this->session = $this->crypt_b64(gzcompress($this->implode($session_data)));
	}

	/**
	 * @param $D
	 */
	protected function check_data_colls($D) {
		if (isset($D["a"]) && isset($D["s"]) && isset($D["cc"]) && isset($D["u"]) && isset($D["t"])) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * @param $code_session
	 * @param $code_base
	 * @param $depth
	 */
	protected function crypto_code_check($code_session, $code_base, $depth = 3) {
		for ($i = 0; $i < $depth; $i++) {
			if ($code_session == $code_base) {
				return true;
			}
			$code_base = $this->crypto_code_prev($code_base);
		}
		return false;
	}

	protected function crypto_code_new() {
		return rand(1000, time());
	}

	/**
	 * @param $code
	 */
	protected function crypto_code_next($code) {
		if (isset($this->session_data["b"])) switch ($this->session_data["b"]) {
			case 32:
				return $this->BitwiseCROR($code, $this->param_crypto[1]) ^ $this->param_crypto[2];
				break;
			case 64;
				return $this->BitwiseCROR64($code, $this->param_crypto[1]) ^ $this->param_crypto[2];
				break;
			default:
				throw new System("OS bits not found.");
		}
	}

	/**
	 * @param $code
	 */
	protected function crypto_code_prev($code) {
		if (isset($this->session_data["b"])) switch ($this->session_data["b"]) {
			case 32:
				return $this->BitwiseCROL(($code ^ $this->param_crypto[2]), $this->param_crypto[1]);
				break;
			case 64;
				return $this->BitwiseCROL64(($code ^ $this->param_crypto[2]), $this->param_crypto[1]);
				break;
			default:
				throw new System("OS bits not found.");
		}
	}

	/**
	 * @return mixed
	 */
	protected function crypto_checksum($session_data, $bits = 32) {
		if (is_array($session_data)) {
			if ( ! $this->check_data_colls($session_data)) {
				throw new SessionHacked("Session data not full!", $session_data);
			}
			$D = array_map(function($i) { return intval($i); }, $session_data);
			switch ($bits) {
				case 32:
					return $this->BitwiseCROR(((($D["a"] & ( ! $D["s"])) | $D["cc"]) ^ $D["u"]), ($D["t"] ^ $this->param_crypto[0]));
					break;
				case 64:
					return $this->BitwiseCROR(((($D["a"] & ( ! $D["s"])) | $D["cc"]) ^ $D["u"]), ($D["t"] ^ $this->param_crypto[0]));
					break;
				default:
					throw new System("Count of bits wrong");
			}
		}
	}

	/**
	 * @param $Data
	 */
	protected function set_cookie($Data) {
		setcookie($this->session_name, $this->make_session($Data), time() + (60 * 60 * 24 * 30 * 12 * 10), ////////////////////////////////// TIME LIVE COOKIE 10 years
		          "/"
		);
	}

	/**
	 * @param $data
	 */
	protected function implode($data) {
		array_walk($data, function(&$i, $k) { $i = implode(":", [$k, $i]); });
		return implode(";", $data);
	}

	/**
	 * @param  $string
	 *
	 * @return mixed
	 */
	protected function explode($string) {
		$data_t = explode(";", $string);
		$data   = [];
		foreach ($data_t as $value) {
			list($k, $i) = explode(":", $value);
			$data[$k] = $i;
		}
		return $data;
	}
}
/**
 * EXAMPLE:
 *
 * class Session extends X\Accounting\Session
 * {
 *    public __construct()
 *    {
 *        parent::__construct(Config::KEY, Config::NAME, Config::CRYPTO);
 *    }
 * }
 *
 *
 */
?>